import asyncio
import enum
import socket
import os

from datetime import datetime
from typing import TextIO

from py_pli.pylib import VUnits
from py_pli.pylib import Measurements
from py_pli.pylib import GlobalVar

from py_pli.pyexception import UrpcFirmwareException

import config_enum.excitationlight_selector_enum as els_config
import config_enum.scan_table_enum as st_config
import config_enum.measurement_unit_enum as meas_config

from meas_services.instrument import InstrumentService

from virtualunits.HAL import HAL
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal
from virtualunits.meas_seq_generator import MeasurementChannel
from virtualunits.meas_seq_generator import IntegratorMode
from virtualunits.meas_seq_generator import AnalogControlMode

from urpc_enum.measurementparameter import MeasurementParameter

from fleming.common.firmware_util import send_gc_msg
from fleming.common.firmware_util import write_gc_msg

from fleming.common.node_io import EEFAnalogInput


instrument: InstrumentService = Measurements.instance.instrument

hal_unit: HAL = VUnits.instance.hal
meas_unit = hal_unit.measurementUnit
els_unit = hal_unit.excitationLightSelector
fms_unit = hal_unit.filterModuleSlider
fm_unit = hal_unit.focusMover
st_unit = hal_unit.scan_table
as1_unit = hal_unit.detectorApertureSlider1
as2_unit = hal_unit.detectorApertureSlider2


async def pmt_analog_adjust_trf_init(filter_module=4001, plate="'96 Test Plate'"):
    """
    Initialize the instrument for TRF based analog adjustment of the PMTs.

    filter_module:
        The filter module ID or for filter without barcode the slot number 1 to 6.
    plate:
        The plate name including the quotes as defined in config/data/3_Inventory-Data.jason
    """

    filter_module = filter_module if (filter_module != '') else 4001
    plate = plate if (plate != '') else "'96 Test Plate'"

    await send_gc_msg(f"Initializing Instrument")
    await instrument.InitializeInstrument()

    await send_gc_msg(f"Setup TRF Measurement")
    st_unit.SetPlateType(plate)
    st_unit.SetCurrentMeasPosition(st_config.GC_Params.FBDTop_TopLeftCorner)
    
    await els_unit.GotoPosition(els_config.Positions.Flash2)
    if (filter_module >= 1) and (filter_module <= 6):
        await fms_unit.SelectModule(filter_module - 1)
    else:
        await fms_unit.SelectModuleWithId(filter_module)

    await fm_unit.GotoFocusHeight(height_mm=7, top=True)

    return f"pmt_analog_adjust_trf_init() done"


async def pmt_analog_adjust_trf(well_col=1, well_row=6, as_start=0.5, as_stop=4.4, as_step=0.01, delay_us=500, window_us=200, flashes=100, flash_mode=1, pre_flashes=100):
    """
    TRF based adjustment of the PMTs analog scaling factors.

    well_col:
        The column of the well to measure.
    well_row:
        The row of the well to measure.
    as_start:
        The start position of the aperture slider scan.
    as_stop:
        The stop position of the aperture slider scan.
    as_step:
        The position increment for the aperture slider scan.
    delay_us:
        The delay of the TRF measurement in µs.
    window_us:
        The measurement window of the TRF measurement in µs.
    flashes:
        The number of flashes per measurement.
    flash_mode:
        0 = High Speed, 1 = High Power
    pre_flashes:
        The number of flashes preceding the measurement.
    """

    well_col = int(well_col) if (well_col != '') else 1
    well_row = int(well_row) if (well_row != '') else 6
    as_start = float(as_start) if (as_start != '') else 0.5
    as_stop = float(as_stop) if (as_stop != '') else 4.4
    as_step = float(as_step) if (as_step != '') else 0.01
    delay_us = int(delay_us) if (delay_us != '') else 500
    window_us = int(window_us) if (window_us != '') else 200
    flashes = int(flashes) if (flashes != '') else 100
    flash_mode = int(flash_mode) if (flash_mode != '') else 1
    pre_flashes = int(pre_flashes) if (pre_flashes != '') else 100
    
    GlobalVar.set_stop_gc(False)

    # The limit per flash for the analog counting equivalent calculation.
    # Only measurements with an anlog low results below (alrs_limit * flashes) are used.
    # Uses the AnalogLimit_PMT1 with 100 mV extra overlap.
    if (meas_unit.AnalogLimit_PMT1 <= 0) or (meas_unit.AnalogLimit_PMT2 <= 0):
        raise ValueError(f"AnalogLimit_PMT1 and AnalogLimit_PMT2 must have valid values")
    pmt1_alrs_limit = meas_unit.AnalogLimit_PMT1 + 1310
    pmt2_alrs_limit = meas_unit.AnalogLimit_PMT2 + 1310
    await send_gc_msg(f"Analog Low to Counting calculation limit: pmt1 = {pmt1_alrs_limit}, pmt2 = {pmt2_alrs_limit}")
    # The limit per flash for the analog high range scale calculation.
    # Only measurements with an anlog low results below (limit * flashes) are used.
    # Uses 90% of the analog low range with 100 mV offset voltage.
    pmt1_ahrs_limit = round((65536 - 1310) * 0.9)
    pmt2_ahrs_limit = round((65536 - 1310) * 0.9)
    await send_gc_msg(f"Analog High to Low calculation limit: pmt1 = {pmt1_ahrs_limit}, pmt2 = {pmt2_ahrs_limit}")

    pmt1_ppr_s = meas_unit.ppr_pmt1
    pmt2_ppr_s = meas_unit.ppr_pmt2

    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/pmt_analog_adjust_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/pmt_analog_adjust_trf__{instrument}_{timestamp}.csv", 'w') as file:
        await write_gc_msg(file, f"pmt_analog_adjust_trf(well_col={well_col}, well_row={well_row}, as_start={as_start}, as_stop={as_stop}, as_step={as_step}, delay_us={delay_us}, window_us={window_us}, flashes={flashes}, flash_mode={flash_mode}) on {instrument} started at {timestamp}")
        await write_gc_msg(file, f"pos [mm] ; pmt1_cr    ; pmt1_alr   ; pmt1_ahr   ; pmt2_cr    ; pmt2_alr   ; pmt2_ahr   ; ref_alr    ; ref_ahr    ; pmt1_dt    ; pmt2_dt")

        await meas_unit.EnableFlashLampPower(flash_mode != 1)
        await st_unit.MoveToWell(well_col, well_row)
        await as1_unit.Home()
        await as2_unit.Home()

        op_id = 'pmt_analog_adjust_trf'
        meas_unit.ClearOperations()
        await load_pmt_analog_adjust_trf(op_id, delay_us, window_us, flashes, flash_mode, pre_flashes)

        pmt1_cr_alr  = 0
        pmt1_alr_alr = 0
        pmt1_alrs    = 0
        pmt2_cr_alr  = 0
        pmt2_alr_alr = 0
        pmt2_alrs    = 0
        pmt1_alr_ahr = 0
        pmt1_ahr_ahr = 0
        pmt1_ahrs    = 0
        pmt2_alr_ahr = 0
        pmt2_ahr_ahr = 0
        pmt2_ahrs    = 0

        pos_range = [pos / 1e6 for pos in range(round(as_start * 1e6), round((as_stop + as_step) * 1e6), round(as_step * 1e6))]
        for pos in pos_range:
            if GlobalVar.get_stop_gc() is True:
                await meas_unit.EnableFlashLampPower(isLow=True)
                await as1_unit.Home()
                await as2_unit.Home()
                return f"pmt_analog_adjust_trf() stopped by user"

            await as1_unit.Move(pos)
            await as2_unit.Move(pos)
            await meas_unit.ExecuteMeasurement(op_id)
            results = await meas_unit.ReadMeasurementValues(op_id)

            ref_alr  = results[0]
            ref_ahr  = results[1]
            pmt1_cr  = results[2] + (results[3] << 32)
            pmt1_dt  = results[4] + (results[5] << 32)
            pmt1_alr = results[6]
            pmt1_ahr = results[7]
            pmt2_cr  = results[8] + (results[9] << 32)
            pmt2_dt  = results[10] + (results[11] << 32)
            pmt2_alr = results[12]
            pmt2_ahr = results[13]

            # Count Rate Correction
            meas_window_s = window_us * 1e-6 * flashes
            pmt1_cr = round(pmt1_cr / (1 - pmt1_cr * pmt1_ppr_s / meas_window_s))
            pmt2_cr = round(pmt2_cr / (1 - pmt2_cr * pmt2_ppr_s / meas_window_s))

            await write_gc_msg(file, f"{pos:8.3f} ; {pmt1_cr:10d} ; {pmt1_alr:10d} ; {pmt1_ahr:10d} ; {pmt2_cr:10d} ; {pmt2_alr:10d} ; {pmt2_ahr:10d} ; {ref_alr:10d} ; {ref_ahr:10d} ; {pmt1_dt:10d} ; {pmt2_dt:10d}")

            if (pmt1_alr < (pmt1_alrs_limit * flashes)):
                pmt1_cr_alr += pmt1_cr * pmt1_alr
                pmt1_alr_alr += pmt1_alr * pmt1_alr
            if (pmt2_alr < (pmt2_alrs_limit * flashes)):
                pmt2_cr_alr += pmt2_cr * pmt2_alr
                pmt2_alr_alr += pmt2_alr * pmt2_alr
            if (pmt1_alr < (pmt1_ahrs_limit * flashes)):
                pmt1_alr_ahr += pmt1_alr * pmt1_ahr
                pmt1_ahr_ahr += pmt1_ahr * pmt1_ahr
            if (pmt2_alr < (pmt2_ahrs_limit * flashes)):
                pmt2_alr_ahr += pmt2_alr * pmt2_ahr
                pmt2_ahr_ahr += pmt2_ahr * pmt2_ahr

        await write_gc_msg(file, '')
        if (pmt1_alr_alr != 0):
            pmt1_alrs = pmt1_cr_alr / pmt1_alr_alr
        await write_gc_msg(file, f"AnalogCountingEquivalent_PMT1 = {pmt1_alrs}")
        if (pmt2_alr_alr != 0):
            pmt2_alrs = pmt2_cr_alr / pmt2_alr_alr
        await write_gc_msg(file, f"AnalogCountingEquivalent_PMT2 = {pmt2_alrs}")
        if (pmt1_ahr_ahr != 0):
            pmt1_ahrs = pmt1_alr_ahr / pmt1_ahr_ahr
        await write_gc_msg(file, f"AnalogHighRangeScale_PMT1 = {pmt1_ahrs}")
        if (pmt2_ahr_ahr != 0):
            pmt2_ahrs = pmt2_alr_ahr / pmt2_ahr_ahr
        await write_gc_msg(file, f"AnalogHighRangeScale_PMT2 = {pmt2_ahrs}")

    await meas_unit.EnableFlashLampPower(isLow=True)
    await as1_unit.Home()
    await as2_unit.Home()

    return f"pmt_analog_adjust_trf() done"


async def load_pmt_analog_adjust_trf(op_id, delay_us=500, window_us=200, flashes=100, flash_mode=1, pre_flashes=100):
    if (delay_us < 50) or (delay_us > 671088):
        raise ValueError(f"delay_us must be in the range [50, 671088] us")
    if (window_us < 1):
        raise ValueError(f"window_us must be greater or equal to 1 us")
    if (flashes < 1) or (flashes > 65536):
        raise ValueError(f"flashes must be in the range [1, 65536]")
    if (pre_flashes < 0) or (pre_flashes > 65536):
        raise ValueError(f"pre_flashes must be in the range [0, 65536]")

    if (flash_mode != 1):
        flash_arm_time = round(meas_unit.lowPower_flashArmingTime * 100)
        flash_total_time = round(1.0 / meas_unit.lowPowerFrequency * 1e8)
    else:
        flash_arm_time = round(meas_unit.highPower_flashArmingTime * 100)
        flash_total_time = round(1.0 / meas_unit.highPowerFrequency * 1e8)

    window_delay = round(delay_us * 100)

    pre_cnt_window = 100        #   1 us
    conversion_delay = 1200     #  12 us
    switch_delay = 25           # 250 ns
    hv_gate_delay = 2000        #  20 us
    reference_window = 3000     #  30 us
    fixed_range = 2000          #  20 us
    input_gate_delay = 100      #   1 us

    loop_delay = round(flash_total_time - flash_arm_time - (2 * conversion_delay) - switch_delay - window_delay - input_gate_delay - (window_us * 100) - fixed_range)

    window_coarse, window_fine = divmod(window_us, 65536)

    seq_gen = meas_seq_generator()

    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 0)  # ref_alr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 1)  # ref_ahr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 2)  # pmt1_cr_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 3)  # pmt1_cr_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 4)  # pmt1_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 5)  # pmt1_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 6)  # pmt1_alr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 7)  # pmt1_ahr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 8)  # pmt2_cr_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 9)  # pmt2_cr_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # pmt2_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # pmt2_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # pmt2_alr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # pmt2_ahr

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2 | OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

    if pre_flashes > 0:
        seq_gen.Loop(pre_flashes)
        seq_gen.TimerWaitAndRestart(flash_arm_time)
        seq_gen.SetSignals(OutputSignal.Flash)
        seq_gen.TimerWaitAndRestart(flash_total_time - flash_arm_time)
        seq_gen.ResetSignals(OutputSignal.Flash)
        seq_gen.LoopEnd()

    seq_gen.Loop(flashes)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset, pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(flash_arm_time)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset, pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset)
    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)

    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.low_range_reset, pmt1=IntegratorMode.low_range_reset, pmt2=IntegratorMode.low_range_reset)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, pmt1=AnalogControlMode.read_offset, pmt2=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(hv_gate_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_autorange, pmt1=IntegratorMode.integrate_in_low_range, pmt2=IntegratorMode.integrate_in_low_range)
    seq_gen.ResetSignals(OutputSignal.Flash)

    seq_gen.TimerWaitAndRestart(reference_window - hv_gate_delay)
    seq_gen.SetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_with_fixed_range)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, pmt1=AnalogControlMode.read_offset, pmt2=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(window_delay - fixed_range - reference_window)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef)

    seq_gen.TimerWaitAndRestart(input_gate_delay)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if window_coarse > 0:
        seq_gen.Loop(window_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if window_fine > 0:
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)

    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_high_range, pmt2=IntegratorMode.integrate_in_high_range)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=2)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=4)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=8)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=1)

    if (loop_delay > 0):
        seq_gen.TimerWaitAndRestart(loop_delay)

    seq_gen.ResetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)
    seq_gen.LoopEnd()

    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset, pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset, pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, 14)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


async def pmt_analog_adjust_trf_scan(well_col=1, well_row=6, as_start=0.5, as_stop=4.4, as_step=0.01, delay_us=500, window_us=1, window_count=200, flashes=100, flash_mode=1, pre_flashes=100):
    """
    Multi-Window TRF based adjustment of the PMTs analog scaling factors.
    Analog is measured across all windows. Total window time is window_us * window_count.

    well_col:
        The column of the well to measure.
    well_row:
        The row of the well to measure.
    as_start:
        The start position of the aperture slider scan.
    as_stop:
        The stop position of the aperture slider scan.
    as_step:
        The position increment for the aperture slider scan.
    delay_us:
        The delay of the TRF measurement in µs.
    window_us:
        The measurement window of the Multi-Window TRF measurement in µs.
    window_count:
        The number of windows of the Multi-Window TRF measurement.
    flashes:
        The number of flashes per measurement.
    flash_mode:
        0 = High Speed, 1 = High Power
    pre_flashes:
        The number of flashes preceding the measurement.
    """

    well_col = int(well_col) if (well_col != '') else 1
    well_row = int(well_row) if (well_row != '') else 6
    as_start = float(as_start) if (as_start != '') else 0.5
    as_stop = float(as_stop) if (as_stop != '') else 4.4
    as_step = float(as_step) if (as_step != '') else 0.01
    delay_us = int(delay_us) if (delay_us != '') else 500
    window_us = int(window_us) if (window_us != '') else 1
    window_count = int(window_count) if (window_count != '') else 200
    flashes = int(flashes) if (flashes != '') else 100
    flash_mode = int(flash_mode) if (flash_mode != '') else 1
    pre_flashes = int(pre_flashes) if (pre_flashes != '') else 100
    
    GlobalVar.set_stop_gc(False)

    # The limit per flash for the analog counting equivalent calculation.
    # Only measurements with an anlog low results below (alrs_limit * flashes) are used.
    # Uses the AnalogLimit_PMT1 with 100 mV extra overlap.
    if (meas_unit.AnalogLimit_PMT1 <= 0) or (meas_unit.AnalogLimit_PMT2 <= 0):
        raise ValueError(f"AnalogLimit_PMT1 and AnalogLimit_PMT2 must have valid values")
    pmt1_alrs_limit = meas_unit.AnalogLimit_PMT1 + 1310
    pmt2_alrs_limit = meas_unit.AnalogLimit_PMT2 + 1310
    await send_gc_msg(f"Analog Low to Counting calculation limit: pmt1 = {pmt1_alrs_limit}, pmt2 = {pmt2_alrs_limit}")
    # The limit per flash for the analog high range scale calculation.
    # Only measurements with an anlog low results below (limit * flashes) are used.
    # Uses 90% of the analog low range with 100 mV offset voltage.
    pmt1_ahrs_limit = round((65536 - 1310) * 0.9)
    pmt2_ahrs_limit = round((65536 - 1310) * 0.9)
    await send_gc_msg(f"Analog High to Low calculation limit: pmt1 = {pmt1_ahrs_limit}, pmt2 = {pmt2_ahrs_limit}")

    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/pmt_analog_adjust_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/pmt_analog_adjust_trf_scan__{instrument}_{timestamp}.csv", 'w') as file:
        await write_gc_msg(file, f"pmt_analog_adjust_trf_scan(well_col={well_col}, well_row={well_row}, as_start={as_start}, as_stop={as_stop}, as_step={as_step}, delay_us={delay_us}, window_us={window_us}, window_count={window_count}, flashes={flashes}, flash_mode={flash_mode}) on {instrument} started at {timestamp}")
        await write_gc_msg(file, f"pos [mm] ; pmt1_cr    ; pmt1_alr   ; pmt1_ahr   ; pmt2_cr    ; pmt2_alr   ; pmt2_ahr   ; ref_alr    ; ref_ahr    ; pmt1_dt    ; pmt2_dt")

        await meas_unit.EnableFlashLampPower(flash_mode != 1)
        await st_unit.MoveToWell(well_col, well_row)
        await as1_unit.Home()
        await as2_unit.Home()

        op_id = 'pmt_analog_adjust_trf_scan'
        meas_unit.ClearOperations()
        await load_pmt_analog_adjust_trf_scan(op_id, delay_us, window_us, window_count, flashes, flash_mode, pre_flashes)

        pmt1_cr_alr  = 0
        pmt1_alr_alr = 0
        pmt1_alrs    = 0
        pmt2_cr_alr  = 0
        pmt2_alr_alr = 0
        pmt2_alrs    = 0
        pmt1_alr_ahr = 0
        pmt1_ahr_ahr = 0
        pmt1_ahrs    = 0
        pmt2_alr_ahr = 0
        pmt2_ahr_ahr = 0
        pmt2_ahrs    = 0

        pos_range = [pos / 1e6 for pos in range(round(as_start * 1e6), round((as_stop + as_step) * 1e6), round(as_step * 1e6))]
        for pos in pos_range:
            if GlobalVar.get_stop_gc() is True:
                await meas_unit.EnableFlashLampPower(isLow=True)
                await as1_unit.Home()
                await as2_unit.Home()
                return f"pmt_analog_adjust_trf_scan() stopped by user"

            await as1_unit.Move(pos)
            await as2_unit.Move(pos)
            await meas_unit.ExecuteMeasurement(op_id)
            results = await meas_unit.ReadMeasurementValues(op_id)

            ref_alr  = results[0]
            ref_ahr  = results[1]
            pmt1_cr  = results[2] + (results[3] << 32)
            pmt1_dt  = results[4] + (results[5] << 32)
            pmt1_alr = results[6]
            pmt1_ahr = results[7]
            pmt2_cr  = results[8] + (results[8] << 32)
            pmt2_dt  = results[10] + (results[11] << 32)
            pmt2_alr = results[12]
            pmt2_ahr = results[13]
            
            pmt1_cr_scan = 0
            pmt1_dt_scan = 0
            pmt2_cr_scan = 0
            pmt2_dt_scan = 0
            for i in range(window_count):
                pmt1_cr_scan += results[i * 4 + 14]
                pmt1_dt_scan += results[i * 4 + 15]
                pmt2_cr_scan += results[i * 4 + 16]
                pmt2_dt_scan += results[i * 4 + 17]

            with open(f"{report_dir}/pmt1_cr_scan.txt", mode='a') as pmt1_cr_scan_file:
                pmt1_cr_scan_file.write(f"[{timestamp}];")
                for i in range(window_count):
                    pmt1_cr_scan_file.write(f"{results[i * 4 + 14]};")
                pmt1_cr_scan_file.write(f"\n")

            with open(f"{report_dir}/pmt1_dt_scan.txt", mode='a') as pmt1_dt_scan_file:
                pmt1_dt_scan_file.write(f"[{timestamp}];")
                for i in range(window_count):
                    pmt1_dt_scan_file.write(f"{results[i * 4 + 15]};")
                pmt1_dt_scan_file.write(f"\n")

            await write_gc_msg(file, f"{pos:8.3f} ; {pmt1_cr:10d} ; {pmt1_alr:10d} ; {pmt1_ahr:10d} ; {pmt2_cr:10d} ; {pmt2_alr:10d} ; {pmt2_ahr:10d} ; {ref_alr:10d} ; {ref_ahr:10d} ; {pmt1_dt:10d} ; {pmt2_dt:10d}")

            await send_gc_msg(f"pmt1_cr_scan: {pmt1_cr_scan}/{pmt1_cr}")
            await send_gc_msg(f"pmt1_dt_scan: {pmt1_dt_scan}/{pmt1_dt}")

            if (pmt1_alr < (pmt1_alrs_limit * flashes)):
                pmt1_cr_alr += pmt1_cr * pmt1_alr
                pmt1_alr_alr += pmt1_alr * pmt1_alr
            if (pmt2_alr < (pmt2_alrs_limit * flashes)):
                pmt2_cr_alr += pmt2_cr * pmt2_alr
                pmt2_alr_alr += pmt2_alr * pmt2_alr
            if (pmt1_alr < (pmt1_ahrs_limit * flashes)):
                pmt1_alr_ahr += pmt1_alr * pmt1_ahr
                pmt1_ahr_ahr += pmt1_ahr * pmt1_ahr
            if (pmt2_alr < (pmt2_ahrs_limit * flashes)):
                pmt2_alr_ahr += pmt2_alr * pmt2_ahr
                pmt2_ahr_ahr += pmt2_ahr * pmt2_ahr

        await write_gc_msg(file, '')
        if (pmt1_alr_alr != 0):
            pmt1_alrs = pmt1_cr_alr / pmt1_alr_alr
        await write_gc_msg(file, f"AnalogCountingEquivalent_PMT1 = {pmt1_alrs}")
        if (pmt2_alr_alr != 0):
            pmt2_alrs = pmt2_cr_alr / pmt2_alr_alr
        await write_gc_msg(file, f"AnalogCountingEquivalent_PMT2 = {pmt2_alrs}")
        if (pmt1_ahr_ahr != 0):
            pmt1_ahrs = pmt1_alr_ahr / pmt1_ahr_ahr
        await write_gc_msg(file, f"AnalogHighRangeScale_PMT1 = {pmt1_ahrs}")
        if (pmt2_ahr_ahr != 0):
            pmt2_ahrs = pmt2_alr_ahr / pmt2_ahr_ahr
        await write_gc_msg(file, f"AnalogHighRangeScale_PMT2 = {pmt2_ahrs}")

    await meas_unit.EnableFlashLampPower(isLow=True)
    await as1_unit.Home()
    await as2_unit.Home()

    return f"pmt_analog_adjust_trf_scan() done"


async def load_pmt_analog_adjust_trf_scan(op_id, delay_us=500, window_us=1, window_count=200, flashes=100, flash_mode=1, pre_flashes=100):
    if (delay_us < 50) or (delay_us > 671088):
        raise ValueError(f"delay_us must be in the range [50, 671088] us")
    if (window_us < 1) or (window_us > 65536):
        raise ValueError(f"window_us must be in the range [1, 65536] us")
    if (window_count < 1) or (window_count > 1020):
        raise ValueError(f"window_count must be in the range [1, 1020]")
    if (flashes < 1) or (flashes > 65536):
        raise ValueError(f"flashes must be in the range [1, 65536]")
    if (pre_flashes < 0) or (pre_flashes > 65536):
        raise ValueError(f"flashes must be in the range [0, 65536]")

    if (flash_mode != 1):
        flash_arm_time = round(meas_unit.lowPower_flashArmingTime * 100)
        flash_total_time = round(1.0 / meas_unit.lowPowerFrequency * 1e8)
    else:
        flash_arm_time = round(meas_unit.highPower_flashArmingTime * 100)
        flash_total_time = round(1.0 / meas_unit.highPowerFrequency * 1e8)

    window_delay = round(delay_us * 100)

    pre_cnt_window = 100        #   1 us
    conversion_delay = 1200     #  12 us
    switch_delay = 25           # 250 ns
    hv_gate_delay = 2000        #  20 us
    reference_window = 3000     #  30 us
    fixed_range = 2000          #  20 us
    input_gate_delay = 100      #   1 us

    loop_delay = round(flash_total_time - flash_arm_time - (2 * conversion_delay) - switch_delay - window_delay - input_gate_delay - (window_us * window_count * 100) - fixed_range)

    seq_gen = meas_seq_generator()

    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 0)  # ref_alr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 1)  # ref_ahr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 2)  # pmt1_cr_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 3)  # pmt1_cr_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 4)  # pmt1_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 5)  # pmt1_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 6)  # pmt1_alr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 7)  # pmt1_ahr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 8)  # pmt2_cr_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 9)  # pmt2_cr_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # pmt2_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # pmt2_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # pmt2_alr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # pmt2_ahr
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=14)
    seq_gen.Loop(window_count)
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=0)    # pmt1_cr
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=1)    # pmt1_dt
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=2)    # pmt2_cr
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=3)    # pmt2_dt
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=4)
    seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2 | OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

    if pre_flashes > 0:
        seq_gen.Loop(pre_flashes)
        seq_gen.TimerWaitAndRestart(flash_arm_time)
        seq_gen.SetSignals(OutputSignal.Flash)
        seq_gen.TimerWaitAndRestart(flash_total_time - flash_arm_time)
        seq_gen.ResetSignals(OutputSignal.Flash)
        seq_gen.LoopEnd()

    seq_gen.Loop(flashes)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset, pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(flash_arm_time)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset, pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset)
    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)

    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.low_range_reset, pmt1=IntegratorMode.low_range_reset, pmt2=IntegratorMode.low_range_reset)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, pmt1=AnalogControlMode.read_offset, pmt2=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(hv_gate_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_autorange, pmt1=IntegratorMode.integrate_in_low_range, pmt2=IntegratorMode.integrate_in_low_range)
    seq_gen.ResetSignals(OutputSignal.Flash)

    seq_gen.TimerWaitAndRestart(reference_window - hv_gate_delay)
    seq_gen.SetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_with_fixed_range)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, pmt1=AnalogControlMode.read_offset, pmt2=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(window_delay - fixed_range - reference_window)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef)

    seq_gen.TimerWaitAndRestart(input_gate_delay)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)

    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=14)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(window_count)

    seq_gen.Loop(window_us)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.LoopEnd()

    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=False, relative=True, resetCounter=False, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=True, relative=True, resetCounter=False, cumulative=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=False, relative=True, resetCounter=False, cumulative=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=True, relative=True, resetCounter=False, cumulative=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=4)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=2)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=4)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=8)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=10)
    seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)

    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_high_range, pmt2=IntegratorMode.integrate_in_high_range)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=1)

    if (loop_delay > 0):
        seq_gen.TimerWaitAndRestart(loop_delay)

    seq_gen.ResetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)
    seq_gen.LoopEnd()

    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset, pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset, pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, (14 + window_count * 4))
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


async def pmt_analog_adjust_lum_init(filter_module=3001, plate="'384 OptiPlate (Black)'"):
    """
    Initialize the instrument for LUM based analog adjustment of the PMTs.

    filter_module:
        The filter module ID or for filter without barcode the slot number 1 to 6.
    plate:
        The plate name including the quotes as defined in config/data/3_Inventory-Data.jason
    """

    filter_module = filter_module if (filter_module != '') else 3001
    plate = plate if (plate != '') else "'384 OptiPlate (Black)'" #"'96 Test Plate'"

    await send_gc_msg(f"Initializing Instrument")
    await instrument.InitializeInstrument()

    await send_gc_msg(f"Setup LUM Measurement")
    st_unit.SetPlateType(plate)
    st_unit.SetCurrentMeasPosition(st_config.GC_Params.FBDTop_TopLeftCorner)
    
    if (filter_module >= 1) and (filter_module <= 6):
        await fms_unit.SelectModule(filter_module - 1)
    else:
        await fms_unit.SelectModuleWithId(filter_module)

    await fm_unit.GotoFocusHeight(height_mm=7, top=True)

    return f"pmt_analog_adjust_lum_init() done"


async def pmt_analog_adjust_lum(well_col=11, well_row=6, as_start=0.5, as_stop=4.4, as_step=0.01, window_ms=5.0, window_count=100):
    """
    LUM based adjustment of the PMTs analog scaling factors.

    well_col:
        The column of the well to measure.
    well_row:
        The row of the well to measure.
    as_start:
        The start position of the aperture slider scan.
    as_stop:
        The stop position of the aperture slider scan.
    as_step:
        The position increment for the aperture slider scan.
    window_ms:
        The measurement window of the LUM measurement in ms.
    window_count:
        The number of windows per measurement.
    """

    well_col = int(well_col) if (well_col != '') else 11
    well_row = int(well_row) if (well_row != '') else 6
    as_start = float(as_start) if (as_start != '') else 0.5
    as_stop = float(as_stop) if (as_stop != '') else 4.4
    as_step = float(as_step) if (as_step != '') else 0.01
    window_ms = float(window_ms) if (window_ms != '') else 5.0
    window_count = int(window_count) if (window_count != '') else 100
    
    GlobalVar.set_stop_gc(False)

    # The limit per flash for the analog counting equivalent calculation.
    # Only measurements with an anlog low results below (limit * flashes) are used.
    # Uses 90% of the analog low range with 100 mV offset voltage.
    pmt1_alrs_limit = round((65536 - 1310) * 0.9)
    pmt2_alrs_limit = round((65536 - 1310) * 0.9)
    await send_gc_msg(f"Analog Low to Counting calculation limit: pmt1 = {pmt1_alrs_limit}, pmt2 = {pmt2_alrs_limit}")
    # The limit per flash for the analog high range scale calculation.
    # Only measurements with an anlog low results below (limit * flashes) are used.
    # Uses 90% of the analog low range with 100 mV offset voltage.
    pmt1_ahrs_limit = round((65536 - 1310) * 0.9)
    pmt2_ahrs_limit = round((65536 - 1310) * 0.9)
    await send_gc_msg(f"Analog High to Low calculation limit: pmt1 = {pmt1_ahrs_limit}, pmt2 = {pmt2_ahrs_limit}")

    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/pmt_analog_adjust_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/pmt_analog_adjust_lum__{instrument}_{timestamp}.csv", 'w') as file:
        await write_gc_msg(file, f"pmt_analog_adjust_lum(well_col={well_col}, well_row={well_row}, as_start={as_start}, as_stop={as_stop}, as_step={as_step}, window_ms={window_ms}, window_count={window_count}) on {instrument} started at {timestamp}")
        await write_gc_msg(file, f"pos [mm] ; pmt1_cr    ; pmt1_alr   ; pmt1_ahr   ; pmt2_cr    ; pmt2_alr   ; pmt2_ahr    ; pmt1_dt    ; pmt2_dt")

        await st_unit.MoveToWell(well_col, well_row)
        await as1_unit.Home()
        await as2_unit.Home()

        op_id = 'pmt_analog_adjust_lum'
        meas_unit.ClearOperations()
        await load_pmt_analog_adjust_lum(op_id, window_ms, window_count)

        pmt1_cr_alr  = 0
        pmt1_alr_alr = 0
        pmt1_alrs    = 0
        pmt2_cr_alr  = 0
        pmt2_alr_alr = 0
        pmt2_alrs    = 0
        pmt1_alr_ahr = 0
        pmt1_ahr_ahr = 0
        pmt1_ahrs    = 0
        pmt2_alr_ahr = 0
        pmt2_ahr_ahr = 0
        pmt2_ahrs    = 0

        pos_range = [pos / 1e6 for pos in range(round(as_start * 1e6), round((as_stop + as_step) * 1e6), round(as_step * 1e6))]
        for pos in pos_range:
            if GlobalVar.get_stop_gc() is True:
                return f"pmt_analog_adjust_lum() stopped by user"

            await as1_unit.Move(pos)
            await as2_unit.Move(pos)
            await meas_unit.ExecuteMeasurement(op_id)
            results = await meas_unit.ReadMeasurementValues(op_id)

            pmt1_cr  = results[0] + (results[1] << 32)
            pmt1_dt  = results[2] + (results[3] << 32)
            pmt1_alr = results[4]
            pmt1_ahr = results[5]
            pmt2_cr  = results[6] + (results[7] << 32)
            pmt2_dt  = results[8] + (results[9] << 32)
            pmt2_alr = results[10]
            pmt2_ahr = results[11]

            await write_gc_msg(file, f"{pos:8.3f} ; {pmt1_cr:10d} ; {pmt1_alr:10d} ; {pmt1_ahr:10d} ; {pmt2_cr:10d} ; {pmt2_alr:10d} ; {pmt2_ahr:10d} ; {pmt1_dt:10d} ; {pmt2_dt:10d}")

            if (pmt1_alr < (pmt1_alrs_limit * window_count)):
                pmt1_cr_alr += pmt1_cr * pmt1_alr
                pmt1_alr_alr += pmt1_alr * pmt1_alr
            if (pmt2_alr < (pmt2_alrs_limit * window_count)):
                pmt2_cr_alr += pmt2_cr * pmt2_alr
                pmt2_alr_alr += pmt2_alr * pmt2_alr
            if (pmt1_alr < (pmt1_ahrs_limit * window_count)):
                pmt1_alr_ahr += pmt1_alr * pmt1_ahr
                pmt1_ahr_ahr += pmt1_ahr * pmt1_ahr
            if (pmt2_alr < (pmt2_ahrs_limit * window_count)):
                pmt2_alr_ahr += pmt2_alr * pmt2_ahr
                pmt2_ahr_ahr += pmt2_ahr * pmt2_ahr

        await write_gc_msg(file, '')
        if (pmt1_alr_alr != 0):
            pmt1_alrs = pmt1_cr_alr / pmt1_alr_alr
        await write_gc_msg(file, f"AnalogCountingEquivalent_PMT1 = {pmt1_alrs}")
        if (pmt2_alr_alr != 0):
            pmt2_alrs = pmt2_cr_alr / pmt2_alr_alr
        await write_gc_msg(file, f"AnalogCountingEquivalent_PMT2 = {pmt2_alrs}")
        if (pmt1_ahr_ahr != 0):
            pmt1_ahrs = pmt1_alr_ahr / pmt1_ahr_ahr
        await write_gc_msg(file, f"AnalogHighRangeScale_PMT1 = {pmt1_ahrs}")
        if (pmt2_ahr_ahr != 0):
            pmt2_ahrs = pmt2_alr_ahr / pmt2_ahr_ahr
        await write_gc_msg(file, f"AnalogHighRangeScale_PMT2 = {pmt2_ahrs}")

    await as1_unit.Home()
    await as2_unit.Home()

    return f"pmt_analog_adjust_lum() done"


async def load_pmt_analog_adjust_lum(op_id, window_ms=5.0, window_count=100):
    if (window_ms < 0.001):
        raise ValueError(f"window_ms must be greater or equal to 0.001 ms")
    if (window_count < 1) or (window_count > 65536):
        raise ValueError(f"flashes must be in the range [1, 65536]")

    window_us = round(window_ms * 1000)

    window_coarse, window_fine = divmod(window_us, 65536)

    full_reset_delay = 40000    # 400 us
    pre_cnt_window = 100        #   1 us
    conversion_delay = 1200     #  12 us
    switch_delay = 25           # 250 ns
    fixed_range = 2000          #  20 us
    reset_switch_delay = 2000   #  20 us
    input_gate_delay = 100      #   1 us

    seq_gen = meas_seq_generator()

    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 0)  # pmt1_cr_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 1)  # pmt1_cr_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 2)  # pmt1_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 3)  # pmt1_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 4)  # pmt1_alr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 5)  # pmt1_ahr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 6)  # pmt2_cr_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 7)  # pmt2_cr_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 8)  # pmt2_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 9)  # pmt2_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # pmt2_alr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # pmt2_ahr

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.SetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

    seq_gen.Loop(window_count)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset)
    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)

    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.low_range_reset, pmt2=IntegratorMode.low_range_reset)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.read_offset, pmt2=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(reset_switch_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_low_range, pmt2=IntegratorMode.integrate_in_low_range)

    seq_gen.TimerWaitAndRestart(input_gate_delay)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if window_coarse > 0:
        seq_gen.Loop(window_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if window_fine > 0:
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.read_offset, pmt2=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_high_range, pmt2=IntegratorMode.integrate_in_high_range)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=2)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=6)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=8)

    seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, 14)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


async def pmt_dead_time_adjust_lum(well_col=11, well_row=6, window_ms=1000.0, iterations=100, required_cps=100e3, as1_pos=None, as2_pos=None):
    """
    LUM based adjustment of the PMTs dead time.

    well_col:
        The column of the well to measure.
    well_row:
        The row of the well to measure.
    window_ms:
        The measurement window of the LUM measurement in ms.
    iterations:
        The number of iterations of this measurement.
    required_cps:
        The required signal in counts per second.
    as1_pos:
        The aperture slider position for PMT1 in mm. If not set it will be adjusted for required_cps.
    as2_pos:
        The aperture slider position for PMT2 in mm. If not set it will be adjusted for required_cps.
    """

    well_col = int(well_col) if (well_col != '') else 11
    well_row = int(well_row) if (well_row != '') else 6
    window_ms = float(window_ms) if (window_ms != '') else 1000.0
    iterations = int(iterations) if (iterations != '') else 100
    required_cps = int(required_cps) if (required_cps != '') else 100e3
    as1_pos = float(as1_pos) if (as1_pos != '') else None
    as2_pos = float(as2_pos) if (as2_pos != '') else None
    
    GlobalVar.set_stop_gc(False)

    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/pmt_analog_adjust_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/pmt_dead_time_adjust_lum__{instrument}_{timestamp}.csv", 'w') as file:
        await write_gc_msg(file, f"pmt_dead_time_adjust_lum(well_col={well_col}, well_row={well_row}, window_ms={window_ms}, iterations={iterations}, required_cps={required_cps}, as1_pos={as1_pos}, as2_pos={as2_pos}) on {instrument} started at {timestamp}")

        await st_unit.MoveToWell(well_col, well_row)
        await as1_unit.Home()
        await as2_unit.Home()

        as1_pos, as2_pos = await find_as_pos_lum(required_cps=required_cps, as_start=1.0, as_stop=4.4, as_step=0.01, window_ms=10.0, as1_pos=as1_pos, as2_pos=as2_pos)
        await as1_unit.Move(as1_pos)
        await as2_unit.Move(as2_pos)

        await write_gc_msg(file, f"as1_pos = {as1_pos:.3f}, as2_pos = {as2_pos:.3f}")

        op_id = 'pmt_dead_time_adjust_lum'
        meas_unit.ClearOperations()
        await load_pmt_deat_time_adjust_lum(op_id, window_ms)

        await write_gc_msg(file, f"iteration ; pmt1_cr     ; pmt1_dt     ; pmt1_t [ns] ; pmt2_cr     ; pmt2_dt     ; pmt2_t [ns]")

        for i in range(iterations):
            if GlobalVar.get_stop_gc() is True:
                return f"pmt_dead_time_adjust_lum() stopped by user"

            await meas_unit.ExecuteMeasurement(op_id)
            results = await meas_unit.ReadMeasurementValues(op_id)

            pmt1_cr = results[0] + (results[1] << 32)
            pmt1_dt = results[2] + (results[3] << 32)
            pmt2_cr = results[6] + (results[7] << 32)
            pmt2_dt = results[8] + (results[9] << 32)

            pmt1_t = (pmt1_dt / 1.2e9 / pmt1_cr) * 1e9 if (pmt1_cr > 0) else 0.0
            pmt2_t = (pmt2_dt / 1.2e9 / pmt2_cr) * 1e9 if (pmt2_cr > 0) else 0.0

            await write_gc_msg(file, f"{(i + 1):9d} ; {pmt1_cr:11d} ; {pmt1_dt:11d} ; {pmt1_t:11.6f} ; {pmt2_cr:11d} ; {pmt2_dt:11d} ; {pmt2_t:11.6f}")

    await as1_unit.Home()
    await as2_unit.Home()

    return f"pmt_dead_time_adjust_lum() done"


async def pmt_dead_time_dl_scan(well_col=11, well_row=6, window_ms=1000.0, dl_start=0.0, dl_stop=1.0, dl_step=0.001, required_cps=100e3, as1_pos=None, as2_pos=None):
    """
    Discriminator Level scan measuring the PMTs dead time.

    well_col:
        The column of the well to measure.
    well_row:
        The row of the well to measure.
    window_ms:
        The measurement window of the LUM measurement in ms.
    dl_start:
        The start value of the discriminator level scan.
    dl_stop:
        The stop value of the discriminator level scan.
    dl_step:
        The step value of the discriminator level scan.
    required_cps:
        The required signal in counts per second.
    as1_pos:
        The aperture slider position for PMT1 in mm. If not set it will be adjusted for required_cps.
    as2_pos:
        The aperture slider position for PMT2 in mm. If not set it will be adjusted for required_cps.
    """

    well_col = int(well_col) if (well_col != '') else 11
    well_row = int(well_row) if (well_row != '') else 6
    window_ms = float(window_ms) if (window_ms != '') else 1000.0
    dl_start = float(dl_start) if (dl_start != '') else 0.0
    dl_stop = float(dl_stop) if (dl_stop != '') else 1.0
    dl_step = float(dl_step) if (dl_step != '') else 0.001
    required_cps = int(required_cps) if (required_cps != '') else 100e3
    as1_pos = float(as1_pos) if (as1_pos != '') else None
    as2_pos = float(as2_pos) if (as2_pos != '') else None
    
    GlobalVar.set_stop_gc(False)

    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/pmt_analog_adjust_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/pmt_dead_time_dl_scan__{instrument}_{timestamp}.csv", 'w') as file:
        await write_gc_msg(file, f"pmt_dead_time_dl_scan(well_col={well_col}, well_row={well_row}, window_ms={window_ms}, dl_start={dl_start}, dl_stop={dl_stop}, dl_step={dl_step}, required_cps={required_cps}, as1_pos={as1_pos}, as2_pos={as2_pos}) on {instrument} started at {timestamp}")

        await st_unit.MoveToWell(well_col, well_row)
        await as1_unit.Home()
        await as2_unit.Home()

        if (as1_pos is None) or (as2_pos is None):
            as1_pos, as2_pos = await find_as_pos_lum(required_cps=required_cps, as_start=-1.0, as_stop=-4.9, as_step=-0.01, window_ms=10.0, as1_pos=as1_pos, as2_pos=as2_pos)
        await as1_unit.Move(as1_pos)
        await as2_unit.Move(as2_pos)

        await write_gc_msg(file, f"as1_pos = {as1_pos:.3f}, as2_pos = {as2_pos:.3f}")

        op_id = 'pmt_dead_time_dl_scan'
        meas_unit.ClearOperations()
        await load_pmt_deat_time_adjust_lum(op_id, window_ms)

        await write_gc_msg(file, f"dl          ; pmt1_cr     ; pmt1_dt     ; pmt1_t [ns] ; pmt2_cr     ; pmt2_dt     ; pmt1_t [ns]")

        dl_range = [dl / 1e6 for dl in range(round(dl_start * 1e6), round((dl_stop + dl_step) * 1e6), round(dl_step * 1e6))]
        for dl in dl_range:
            if GlobalVar.get_stop_gc() is True:
                await meas_unit.ReloadConfig()
                await as1_unit.Home()
                await as2_unit.Home()
                return f"pmt_dead_time_dl_scan() stopped by user"

            await meas_unit.endpoint.SetParameter(MeasurementParameter.PMT1DiscriminatorLevel, dl)
            await meas_unit.endpoint.SetParameter(MeasurementParameter.PMT2DiscriminatorLevel, dl)
            
            await meas_unit.ExecuteMeasurement(op_id)
            results = await meas_unit.ReadMeasurementValues(op_id)

            pmt1_cr = results[0] + (results[1] << 32)
            pmt1_dt = results[2] + (results[3] << 32)
            pmt2_cr = results[6] + (results[7] << 32)
            pmt2_dt = results[8] + (results[9] << 32)

            pmt1_t = (pmt1_dt / 1.2e9 / pmt1_cr) * 1e9 if (pmt1_cr > 0) else 0.0
            pmt2_t = (pmt2_dt / 1.2e9 / pmt2_cr) * 1e9 if (pmt2_cr > 0) else 0.0

            await write_gc_msg(file, f"{dl:11.3f} ; {pmt1_cr:11d} ; {pmt1_dt:11d} ; {pmt1_t:11.6f} ; {pmt2_cr:11d} ; {pmt2_dt:11d} ; {pmt2_t:11.6f}")

    await meas_unit.ReloadConfig()
    await as1_unit.Home()
    await as2_unit.Home()

    return f"pmt_dead_time_dl_scan() done"


async def load_pmt_deat_time_adjust_lum(op_id, window_ms=100.0):
    if (window_ms < 0.001):
        raise ValueError(f"window_ms must be greater or equal to 0.001 ms")

    window_us = round(window_ms * 1000)

    window_coarse, window_fine = divmod(window_us, 65536)

    hv_gate_delay = 1000000     #  10 ms
    pre_cnt_window = 100        #   1 us

    seq_gen = meas_seq_generator()

    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 0)  # pmt1_cr_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 1)  # pmt1_cr_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 2)  # pmt1_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 3)  # pmt1_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 4)  # pmt1_alr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 5)  # pmt1_ahr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 6)  # pmt2_cr_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 7)  # pmt2_cr_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 8)  # pmt2_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 9)  # pmt2_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # pmt2_alr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # pmt2_ahr
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # pmt3_cr_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # pmt3_cr_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # pmt3_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # pmt3_dt_msb

    seq_gen.TimerWaitAndRestart(hv_gate_delay)
    seq_gen.SetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if window_coarse > 0:
        seq_gen.Loop(window_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if window_fine > 0:
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()

    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=2)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=6)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=8)
    seq_gen.GetPulseCounterResult(MeasurementChannel.US_LUM, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=12)
    seq_gen.GetPulseCounterResult(MeasurementChannel.US_LUM, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=14)

    seq_gen.ResetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, 16)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


async def find_as_pos_lum(required_cps=100e3, as_start=1.0, as_stop=4.4, as_step=0.01, window_ms=10.0, as1_pos=None, as2_pos=None):

    await send_gc_msg(f"Adjusting Aperture Slider position for a count rate of {required_cps} CPS")

    op_id = 'find_as_pos_lum'
    meas_unit.ClearOperations()
    await load_pmt_deat_time_adjust_lum(op_id, window_ms)

    pos_range = [pos / 1e6 for pos in range(round(as_start * 1e6), round((as_stop + as_step) * 1e6), round(as_step * 1e6))]
    for pos in pos_range:
        if GlobalVar.get_stop_gc() is True:
            return f"find_as_pos() stopped by user"

        await asyncio.gather(as1_unit.Move(pos), as2_unit.Move(pos))

        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        pmt1_cr = results[0] + (results[1] << 32)
        pmt2_cr = results[6] + (results[7] << 32)

        pmt1_cps = pmt1_cr / window_ms * 1000
        pmt2_cps = pmt2_cr / window_ms * 1000
        
        await send_gc_msg(f"pos = {pos:.3f}, pmt1 = {pmt1_cps:.1f}, pmt2 = {pmt2_cps:.1f}")

        if (as1_pos is None) and (pmt1_cps > required_cps):
            as1_pos = pos - as_step
        if (as2_pos is None) and (pmt2_cps > required_cps):
            as2_pos = pos - as_step
        
        if (as1_pos is not None) and (as2_pos is not None):
            await asyncio.gather(as1_unit.Home(), as2_unit.Home())
            return (as1_pos, as2_pos)

    if (as1_pos is None):
        await send_gc_msg(f"Failed to adjust Aperture Slider 1")
        as1_pos = 0.0
    if (as2_pos is None):
        await send_gc_msg(f"Failed to adjust Aperture Slider 2")
        as2_pos = 0.0
    
    await asyncio.gather(as1_unit.Home(), as2_unit.Home())
    return (as1_pos, as2_pos)


async def pmt_lum_scan(well_col=6, well_row=1, window_us=0.1, window_count=4096, iterations=100, required_cps=1e6, as1_pos=None):
    """
    Multi-Window LUM Scan.

    well_col:
        The column of the well to measure.
    well_row:
        The row of the well to measure.
    window_us:
        The measurement window of the LUM measurement in us.
    window_count:
        The number of measurement windows.
    iterations:
        The number of iterations of this measurement.
    required_cps:
        The required signal in counts per second.
    as1_pos:
        The aperture slider position for PMT1 in mm. If not set it will be adjusted for required_cps.
    """

    well_col = int(well_col) if (well_col != '') else 11
    well_row = int(well_row) if (well_row != '') else 6
    window_us = float(window_us) if (window_us != '') else 0.1
    window_count = int(window_count) if (window_count != '') else 4096
    iterations = int(iterations) if (iterations != '') else 100
    required_cps = int(required_cps) if (required_cps != '') else 100e3
    as1_pos = float(as1_pos) if (as1_pos != '') else None
    
    GlobalVar.set_stop_gc(False)

    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/pmt_analog_adjust_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/pmt_lum_scan__{instrument}_{timestamp}.csv", 'w') as file:
        await write_gc_msg(file, f"pmt_lum_scan(well_col={well_col}, well_row={well_row}, window_us={window_us}, window_count={window_count}, iterations={iterations}, required_cps={required_cps}, as1_pos={as1_pos}) on {instrument} started at {timestamp}")

        await st_unit.MoveToWell(well_col, well_row)
        await as1_unit.Home()
        await as2_unit.Home()

        if (as1_pos is None):
            as1_pos, as2_pos = await find_as_pos_lum(required_cps=required_cps, as_start=-1.0, as_stop=-4.9, as_step=-0.01, window_ms=10.0, as1_pos=as1_pos, as2_pos=0.0)
        await as1_unit.Move(as1_pos)

        await write_gc_msg(file, f"as1_pos = {as1_pos:.3f}, as2_pos = {as2_pos:.3f}")

        op_id = 'pmt_lum_scan'
        meas_unit.ClearOperations()
        await load_pmt_lum_scan(op_id, window_us, window_count)

        results = []

        for i in range(iterations):
            if GlobalVar.get_stop_gc() is True:
                await as1_unit.Home()
                await as2_unit.Home()
                return f"pmt_lum_scan() stopped by user"

            await send_gc_msg(f"Measure Scan #{i + 1:04d}")
            
            await meas_unit.ExecuteMeasurement(op_id)
            results.append(await meas_unit.ReadMeasurementValues(op_id))

        msg = f"t [us]    "
        for i in range(iterations):
            msg += f" ; scan_#{i + 1:04d}"
        await write_gc_msg(file, msg)

        for j in range(window_count):
            msg = f"{(j + 1) * window_us:10.1f}"
            for i in range(iterations):
                msg += f" ; {results[i][j]:10f}"
            await write_gc_msg(file, msg)

    await as1_unit.Home()
    await as2_unit.Home()

    return f"pmt_lum_scan() done"    


async def load_pmt_lum_scan(op_id, window_us=1.0, window_count=4096):
    if (window_us < 0.05) or (window_us > 671088.64):
        raise ValueError(f"window_us must be in the range [0.05, 671088.64] us")
    if (window_count < 1) or (window_count > 4096):
        raise ValueError(f"window_us must be in the range [1, 4096]")

    window = round(window_us * 100)

    hv_gate_delay = 1000000     #  10 ms

    seq_gen = meas_seq_generator()

    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(window_count)
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=0)    # pmt1_cr
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()

    seq_gen.TimerWaitAndRestart(hv_gate_delay)
    seq_gen.SetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

    seq_gen.TimerWaitAndRestart(window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(window_count)
    seq_gen.TimerWaitAndRestart(window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, relative=True, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, window_count)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

